import os
import math
import logging
from datetime import datetime

# 配置日志
log_filename = f"file_splitter_log_{datetime.now().strftime('%Y%m%d_%H%M%S')}.log"
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler(log_filename, encoding='utf-8'),
        logging.StreamHandler() # 同时输出到控制台
    ]
)
logger = logging.getLogger(__name__)

def convert_mb_to_bytes(mb):
    """将兆字节转换为字节"""
    return mb * 1024 * 1024

def split_text_file(file_path, max_bytes_per_part):
    """
    切分单个文本文件。
    :param file_path: 待切分的文本文件完整路径。
    :param max_bytes_per_part: 每部分的最大字节数限制。
    :return: 成功切分后创建的新文件路径列表，如果失败则返回空列表。
    """
    original_filename = os.path.basename(file_path)
    base_name, ext = os.path.splitext(original_filename)
    output_dir = os.path.dirname(file_path)

    logger.info(f"  开始切分文件: '{original_filename}'")

    try:
        # 读取所有行。注意：对于极大的文件，这可能会消耗大量内存。
        # 如果遇到内存问题，需要改为逐行读取并动态写入的逻辑。
        with open(file_path, 'r', encoding='utf-8', errors='ignore') as f:
            all_lines = f.readlines()
        
        original_size_bytes = os.path.getsize(file_path)
        total_lines = len(all_lines)

        # 计算最少需要切分成多少份才能满足每份大小限制
        # N_min_parts = ceil(原始文件大小 / 每份最大限制)
        N_min_parts = math.ceil(original_size_bytes / max_bytes_per_part)
        
        # 确保切分份数至少为3，同时满足大小限制
        # 例如：如果原始文件100MB，限制50MB，N_min_parts=2。但要求3+，所以num_parts=3。
        #       每份33.3MB，符合限制。
        # 例如：如果原始文件250MB，限制50MB，N_min_parts=5。num_parts=5。
        #       每份50MB，符合限制。
        num_parts = max(3, N_min_parts)

        logger.info(f"    原始大小: {original_size_bytes / (1024*1024):.2f} MB, "
                    f"每份最大限制: {max_bytes_per_part / (1024*1024):.2f} MB")
        logger.info(f"    计划切分成 {num_parts} 份。")

        # 计算每份大致应包含的行数
        lines_per_part = math.ceil(total_lines / num_parts)

        created_files = []
        start_line_idx = 0
        for i in range(num_parts):
            end_line_idx = min(start_line_idx + lines_per_part, total_lines)
            part_lines = all_lines[start_line_idx:end_line_idx]

            if not part_lines: # 避免创建空文件（如果行数不足以填满所有部分）
                break

            # 生成新文件名：原文件名 (排序数字).txt
            new_part_filename = f"{base_name} ({i+1}){ext}"
            new_part_path = os.path.join(output_dir, new_part_filename)

            with open(new_part_path, 'w', encoding='utf-8') as out_f:
                out_f.writelines(part_lines)
            
            # 验证新文件大小是否超出限制
            actual_part_size = os.path.getsize(new_part_path)
            if actual_part_size > max_bytes_per_part:
                logger.warning(f"  WARNING: 切分后的文件 '{new_part_filename}' ({actual_part_size / (1024*1024):.2f} MB) "
                               f"超出了最大允许大小 ({max_bytes_per_part / (1024*1024):.2f} MB)。"
                               f"这可能发生在单行文本非常长的情况下。请手动检查。")
            else:
                logger.info(f"    创建文件: '{new_part_filename}' (大小: {actual_part_size / (1024*1024):.2f} MB)")

            created_files.append(new_part_path)
            start_line_idx = end_line_idx
        
        # 如果成功创建了至少一个新文件，则删除原始文件
        if created_files:
            os.remove(file_path)
            logger.info(f"  原始文件 '{original_filename}' 已成功切分并删除。")
            return created_files
        else:
            logger.warning(f"  文件 '{original_filename}' 未能成功切分，保留原文件。")
            return []

    except FileNotFoundError:
        logger.error(f"文件未找到: {file_path}")
        return []
    except Exception as e:
        logger.error(f"处理文件 '{original_filename}' 时发生错误: {e}", exc_info=True)
        return []

def process_directory_for_splitting(root_dir, max_file_size_mb):
    """
    遍历指定目录及其子目录，切分符合条件的文本文件。
    """
    max_bytes_per_part = convert_mb_to_bytes(max_file_size_mb)
    split_threshold_bytes = max_bytes_per_part * 0.96 # 96% 的限制大小

    logger.info(f"开始扫描目录: {root_dir}")
    logger.info(f"最大允许文件大小: {max_file_size_mb:.2f} MB")
    logger.info(f"切分阈值 (大于此大小的文件将被切分): {split_threshold_bytes / (1024*1024):.2f} MB (即 {max_file_size_mb * 0.96:.2f} MB)")

    total_files_scanned = 0
    files_split_count = 0
    files_kept_count = 0
    files_skipped_count = 0

    for dirpath, dirnames, filenames in os.walk(root_dir):
        for filename in filenames:
            total_files_scanned += 1
            full_path = os.path.join(dirpath, filename)

            if not filename.lower().endswith('.txt'):
                logger.debug(f"跳过 (非txt文件): {filename}")
                files_skipped_count += 1
                continue

            try:
                file_size_bytes = os.path.getsize(full_path)
                
                if file_size_bytes > split_threshold_bytes:
                    logger.info(f"\n文件 '{filename}' (大小: {file_size_bytes / (1024*1024):.2f} MB) 超过切分阈值，准备切分。")
                    created_parts = split_text_file(full_path, max_bytes_per_part)
                    if created_parts:
                        files_split_count += 1
                    else:
                        files_kept_count += 1 # 切分失败，原文件保留
                else:
                    logger.debug(f"文件 '{filename}' (大小: {file_size_bytes / (1024*1024):.2f} MB) 未达到切分阈值，保留。")
                    files_kept_count += 1

            except FileNotFoundError:
                logger.warning(f"文件 '{full_path}' 在扫描时不存在，跳过。")
                files_skipped_count += 1
            except Exception as e:
                logger.error(f"处理文件 '{full_path}' 时发生未知错误: {e}", exc_info=True)
                files_skipped_count += 1

    logger.info("\n========== 处理总结 ==========")
    logger.info(f"总共扫描文件: {total_files_scanned}")
    logger.info(f"已切分的文件数量: {files_split_count}")
    logger.info(f"已保留的文件数量 (未达到切分阈值或切分失败): {files_kept_count}")
    logger.info(f"已跳过的非 .txt 文件数量: {files_skipped_count}")
    logger.info("================================")
    logger.info(f"详细日志已保存到: {log_filename}")


if __name__ == "__main__":
    root_directory = input("请输入要处理的根目录路径 (例如: D:\\MyFiles): ")
    
    if not os.path.isdir(root_directory):
        logger.error(f"输入的路径 '{root_directory}' 不是一个有效的目录。请检查。")
    else:
        while True:
            try:
                max_size_mb_input = input("请输入最大允许的文本文件大小 (MB): ")
                max_file_size_mb = float(max_size_mb_input)
                if max_file_size_mb <= 0:
                    print("文件大小必须是正数。")
                    continue
                break
            except ValueError:
                print("无效输入，请输入一个数字。")

        process_directory_for_splitting(root_directory, max_file_size_mb)
    
    input("\n程序运行完毕，请查看日志文件。按 Enter 键退出...")

